"use client";

import Link from "next/link";
import { useRouter } from "next/router";
import { useSession, signOut } from "next-auth/react";
import { useSidebar } from "@/contexts/SidebarContext";
import { useLanguage } from "@/contexts/LanguageContext";
import { useState, useEffect } from "react";
import {
  Home,
  Ticket,
  Plus,
  Users,
  Database,
  LogOut,
  User,
  DollarSign,
  Settings,
  ChevronLeft,
  ChevronRight,
  Menu,
  X,
  CheckSquare,
  History,
} from "lucide-react";

export default function Sidebar() {
  const { data: session, status } = useSession();
  const router = useRouter();
  const { isCollapsed, setIsCollapsed } = useSidebar();
  const { t } = useLanguage();
  const [isMobileOpen, setIsMobileOpen] = useState(false);

  const isAdmin = session?.user?.role === "ADMIN";
  const isTechnician = session?.user?.role === "TECHNICIAN";
  const iconSize = isCollapsed ? "w-6 h-6" : "w-5 h-5";

  // prevent scroll behind mobile menu
  useEffect(() => {
    document.body.style.overflow = isMobileOpen ? "hidden" : "auto";
  }, [isMobileOpen]);

  const isActive = (path: string) =>
    router.pathname === path || router.pathname.startsWith(path + "/");

  const handleSignOut = () => signOut({ callbackUrl: "/login" });

  const navigationLinks = [
    {
      href: "/",
      label: t("sidebar.dashboard"),
      icon: <Home className={iconSize} />,
    },
    {
      href: "/tickets",
      label: t("sidebar.tickets"),
      icon: <Ticket className={iconSize} />,
    },
    ...(isAdmin || isTechnician
      ? [
          {
            href: "/tasks",
            label: t("sidebar.tasks") || "Tasks",
            icon: <CheckSquare className={iconSize} />,
          },
        ]
      : []),
    ...(isTechnician
      ? [
          {
            href: "/tasks/history",
            label: t("sidebar.taskHistory") || "Task History",
            icon: <History className={iconSize} />,
          },
        ]
      : []),
    ...(isAdmin
      ? [
          {
            href: "/admin/technicians",
            label: t("sidebar.technicians"),
            icon: <Users className={iconSize} />,
          },
          {
            href: "/admin/catalogs",
            label: t("sidebar.catalogs"),
            icon: <Database className={iconSize} />,
          },
          {
            href: "/admin/cost-analysis",
            label: t("sidebar.costAnalysis"),
            icon: <DollarSign className={iconSize} />,
          },
          {
            href: "/admin/task-analysis",
            label: t("sidebar.taskAnalysis") || "Task Analysis",
            icon: <CheckSquare className={iconSize} />,
          },
          {
            href: "/admin/roles",
            label: t("sidebar.roles"),
            icon: <Settings className={iconSize} />,
          },
        ]
      : []),
  ];

  const NavLink = ({
    href,
    label,
    icon,
  }: {
    href: string;
    label: string;
    icon: React.ReactNode;
  }) => {
    const active = isActive(href);
    const base =
      "flex items-center gap-3 px-3 pr-4 py-2 rounded-lg text-sm font-medium transition-colors";
    const neutral =
      "text-foreground/70 hover:text-foreground hover:bg-sidebar-accent";
    const activeCls = "bg-sidebar-accent text-primary-500";

    return (
      <Link
        href={href}
        onClick={() => setIsMobileOpen(false)}
        aria-current={active ? "page" : undefined}
        className={`${base} ${active ? activeCls : neutral} ${
          isCollapsed ? "justify-center" : ""
        }`}
        title={isCollapsed ? label : undefined}
      >
        {icon}
        {!isCollapsed && <span>{label}</span>}
      </Link>
    );
  };

  return (
    <>
      {/* Overlay */}
      {isMobileOpen && (
        <div
          className='fixed inset-0 bg-black/40 backdrop-blur-sm z-40 lg:hidden transition-opacity'
          onClick={() => setIsMobileOpen(false)}
        />
      )}

      {/* Sidebar */}
      <aside
        className={`fixed left-0 top-0 h-full bg-sidebar border-r border-border flex flex-col justify-between z-50 transition-all duration-300 ease-in-out overflow-y-auto pb-6 ${
          isMobileOpen ? "translate-x-0" : "-translate-x-full"
        } lg:translate-x-0 ${
          isCollapsed ? "w-16" : "w-72 max-w-[80vw]"
        } shadow-lg lg:shadow-none`}
      >
        {/* Close Button (Mobile) */}
        <button
          onClick={() => setIsMobileOpen(false)}
          className='lg:hidden absolute top-4 right-4 p-2 rounded-lg hover:bg-sidebar-accent text-foreground'
          aria-label='Close menu'
        >
          <X className='w-5 h-5' />
        </button>

        {/* Main Scroll Area */}
        <div className='flex flex-col flex-1 overflow-y-auto'>
          {/* Brand */}
          <div
            className='border-b border-border flex justify-center items-center'
            style={{ height: "65px" }}
          >
            <Link
              href='/'
              onClick={() => setIsMobileOpen(false)}
              className={`flex items-center ${
                isCollapsed ? "justify-center" : "gap-3"
              }`}
            >
              <div className='w-9 h-9 bg-primary-500 rounded-lg flex items-center justify-center'>
                <Ticket className='w-5 h-5 text-primary-fore' />
              </div>
              {!isCollapsed && (
                <span className='text-xl font-bold text-foreground'>
                  {session?.user ? t("nav.hospitalPortal") : "Hospital Portal"}
                </span>
              )}
            </Link>
          </div>

          {/* Navigation */}
          <nav
            className={`flex-1 px-2 ${
              isCollapsed ? "space-y-1" : "space-y-2"
            } py-3`}
          >
            {navigationLinks.map((link) => (
              <NavLink key={link.href} {...link} />
            ))}

            {/* Quick Actions */}
            <div className='pt-3 border-t border-border mt-2'>
              <Link
                href='/tickets/new'
                onClick={() => setIsMobileOpen(false)}
                className={`flex items-center gap-3 px-3 pr-4 py-2 rounded-lg text-sm font-medium bg-primary-500 hover:bg-primary-600 transition-colors ${
                  isCollapsed ? "justify-center" : ""
                }`}
                title={isCollapsed ? t("sidebar.newTicket") : undefined}
                style={{ color: "var(--color-primary-fore)" }}
              >
                <Plus className={iconSize} />
                {!isCollapsed && <span>{t("sidebar.newTicket")}</span>}
              </Link>
            </div>
          </nav>

          {/* Collapse Toggle */}
          <div className='px-2 py-2 border-t border-border'>
            <button
              onClick={() => setIsCollapsed(!isCollapsed)}
              className={`w-full flex items-center gap-3 px-3 pr-4 py-2 rounded-lg text-sm font-medium text-foreground/70 hover:text-foreground hover:bg-sidebar-accent transition-colors ${
                isCollapsed ? "justify-center" : ""
              }`}
              title={
                isCollapsed
                  ? t("sidebar.expandSidebar")
                  : t("sidebar.collapseSidebar")
              }
            >
              {isCollapsed ? (
                <ChevronRight className='w-5 h-5' />
              ) : (
                <ChevronLeft className='w-5 h-5' />
              )}
              {!isCollapsed && <span>{t("sidebar.toggleMenu")}</span>}
            </button>
          </div>
        </div>

        {/* Bottom (User + Sign out) */}
        <div className='sticky bottom-0 bg-sidebar border-t border-border p-3 space-y-2'>
          {status === "authenticated" && session?.user ? (
            <>
              <Link
                href='/profile'
                onClick={() => setIsMobileOpen(false)}
                className={`flex items-center gap-3 px-3 pr-4 py-2 rounded-lg text-sm font-medium transition-colors ${
                  isActive("/profile")
                    ? "bg-sidebar-accent text-foreground"
                    : "text-foreground/70 hover:text-foreground hover:bg-sidebar-accent"
                } ${isCollapsed ? "justify-center" : ""}`}
                title={
                  isCollapsed ? session.user.email || undefined : undefined
                }
              >
                <User className={iconSize} />
                {!isCollapsed && (
                  <span className='truncate text-sm'>{session.user.email}</span>
                )}
              </Link>

              <button
                onClick={handleSignOut}
                className={`flex items-center gap-3 px-3 pr-4 py-2 rounded-lg text-sm font-medium text-foreground/70 hover:text-foreground hover:bg-sidebar-accent transition-colors w-full ${
                  isCollapsed ? "justify-center" : ""
                }`}
                title={isCollapsed ? "Sign Out" : undefined}
              >
                <LogOut className={iconSize} />
                {!isCollapsed && <span>{t("sidebar.signOut")}</span>}
              </button>
            </>
          ) : (
            <Link
              href='/login'
              onClick={() => setIsMobileOpen(false)}
              className={`flex items-center gap-2 px-3 pr-4 py-2 rounded-lg text-sm font-medium bg-sidebar-accent text-foreground hover:opacity-90 transition-colors ${
                isCollapsed ? "justify-center" : ""
              }`}
              title={isCollapsed ? "Sign In" : undefined}
            >
              <User className={iconSize} />
              {!isCollapsed && <span>{t("sidebar.signIn")}</span>}
            </Link>
          )}
        </div>
      </aside>

      {/* Floating Mobile Menu Button */}
      <button
        onClick={() => setIsMobileOpen(true)}
        className='fixed bottom-4 left-4 z-40 lg:hidden p-3 bg-primary-500 text-primary-fore rounded-full shadow-lg hover:bg-primary-600 transition-all duration-200'
        aria-label='Open menu'
      >
        <Menu className='w-6 h-6' />
      </button>
    </>
  );
}
